/* 
  Copyright (c) 2002 Gavin E. Crooks <gec@compbio.berkeley.edu>
		     Univ. of California, Berkeley

  Permission is hereby granted, free of charge, to any person obtaining a 
  copy of this software and associated documentation files (the "Software"),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included
  in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN 
  THE SOFTWARE.

  (This is the MIT Open Source License, 
  http://www.opensource.org/licenses/mit-license.html)
*/


/* ********************************************************

 Optimal Posterior

   Find the mean posterior estimate given a mean prior estimate
   and data sampled from a multinomial distribution.

   usage info: optimal_posterior -h

******************************************************** */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <math.h>

#include <unistd.h>

#include "nhypergeometric.h"
#include "dirichlet_ml.h"
#include "die.h"


#define FALSE (0)
#define TRUE (1)
#define MAX_LINE_LENGTH (1024)

#define MAX_K (1000000)

#define VERSION "$Id: optimal_posterior.c,v 1.3 2002/10/07 20:20:34 gec Exp $"


void help( void ) {
  printf("\nOptimal Posterior \n"
         "\n"
         "Find the mean posterior estimate given a mean prior estimate\n"
         "and data sampled from a multinomial distribution.\n"
         "\n"
         "Usage: optimal_posterior <optimal_posterior.testdata\n"
         "\n"
         "Each line of the data file gives the mean prior estimate\n"
         "and observed data for each component of the probability\n"
         "distribution.\n"
         "\n"
         "The mean prior estimate determines the parameters of\n"
         "the natural Dirichlet prior up to an overall scaling \n"
         "constant.  This scale is determined by maximizing the\n"
         "probability of the observed data, which is distributed \n"
         "according to a negative hypergeometric distribution.\n"
         "\n"
         "  Options:\n"
         "\n"
         "  -h           Help: print this help message\n"
         "  -i filename  Read this blocks database (default: stdin)\n" 
         "  -o filename  Output file. (default: stdout)\n"
         "  -v           Verbose: Output extra information \n"
         "  -q           Quite: Not verbose\n"
         "  -V           Version\n"
         "\n"
         );
  exit ( 0 );
}


/* Read data from data file */
void read_data(FILE *in_stream, int * K, double * data, double * mean_prior) {
  char buffer[MAX_LINE_LENGTH];
  int k = 0;

  while(fgets(buffer, MAX_LINE_LENGTH, in_stream)!=NULL) {
    /*Skip for blank and comment lines */
    if( buffer[0] =='#' || buffer[0]=='\n') continue;
    
    if( EOF==sscanf(buffer, "%lf %lf" ,
		    &(mean_prior[k]), &(data[k]) ) ) {
      die("Error reading data ");
    }

    // If mean_prior is zero, then the posterior must be zero,
    // and any observed data would cause a conflict.
    if( mean_prior[k] ==0.0 && data[k] != 0) {
      die("Data error: %lf\t%lf", mean_prior[k], data[k]);
    }

    k++;
  }
  *K = k;
}


/* Print the negative hypergeoemtric probability of 
 * the data for different scale parementers.
 */
void print_ln_prob( const int K, 
                    const double data[], 
                    const double mean_prior[], 
                    FILE * stream) {
  double beta, bot=1, top = 1000000000, ratio = 1.1;
  double * x;
  double * d;
  double lnprob;

  x = (double *) calloc(K, sizeof(double));
  assert(x != 0);
  d = (double *) calloc(K, sizeof(double));
  assert(d != 0);

  fprintf(stream, 
          "# The negative hypergeometric probability of the data\n"
          "# for different scale parameters (beta)\n"
          "\n"
          "# beta\t ln data_prob\n");
   
  for(beta=bot; beta<top; beta*=ratio) {
    dirichlet_to_nhyper(K, data, mean_prior, beta, x, d);
    lnprob = multivariate_negative_hypergeometric_lnpdf(K, x, d);
    fprintf(stream, "%g\t%g\n",beta, lnprob);   
  }
  fprintf(stream, "\n");

  free(x);
  free(d);
}

/* Print the results 
 *   K : distribution dimension
 *   data
 *   mean_prior : 
 *   beta          : Scale parameter
 */
void write_data(FILE * stream, 
		const int K, 
		const double data[],
		const double mean_prior[], 
		const double beta) {
  int k;
  double total_data = 0.0;
  double total_post = 0.0;
  double alpha, post;

  for(k=0; k<K; k++)
    total_data += data[k];

  fprintf(stream,"# mean_prior_estimate    data   data_mean           alpha     mean_posterior_estimate\n");

  for(k=0; k<K; k++) {
    alpha = mean_prior[k] * beta;
    post = (data[k] + alpha) /(total_data + beta);
    total_post += post;
    fprintf(stream, "%14e %14g %14e %14g %14e\n", 
	    mean_prior[k], data[k], data[k]/total_data, alpha, post);
  }

  fprintf(stream, "\n# Total Post: %g\n\n", total_post);
}


/* Calcuate relative entropy between prior and data */
void relative_ent(FILE * stream,
		  const int K, 
		  const double data[],
		  const double mean_prior[] )
{
  int k;
  double total_data = 0.0;
  double ent_PQ=0.0, ent_QP=0.0;
  double p,q;

  for(k=0; k<K; k++) total_data += data[k];

  for(k=0; k<K; k++) {
    p = mean_prior[k];
    q = data[k]/total_data;
    if(p==0.0 || q==0.0) continue;
    ent_PQ += p * log( p/q );
    ent_QP += q * log( q/p );
    /* printf("%g\t%g\t%g\t%g\t%g\n",p,q, p/q, log(p/q), p * log( p/q ) );*/
  }

  fprintf(stream, "\n Relative Entropy:\n  D(data||prior) = %g\n  D(prior||data) = %g\n\n", ent_QP, ent_PQ);

} 


int main(int argc, char *argv[]) {
  int c;

  FILE * in_stream = stdin;
  FILE * out_stream = stdout;
  FILE * verbose_stream = NULL;
  int verbose=FALSE;

  int K;
  double * data;
  double * mean_prior;
  double beta;
  
  while ((c = getopt (argc, argv, "i:o:r:hvqBF")) != -1) {
    switch (c) {
    case 'i' :
      in_stream = fopen(optarg, "r");
      if(in_stream == NULL) die("Cannot open file %s for input.", optarg);
      break;
    case 'o' :
      out_stream = fopen(optarg, "w"); //FIXME
      if(out_stream == NULL) die("Cannot open file %s for output.", optarg);
      break;
    case 'h' :
      help();
      break;
    case 'v' : 
      verbose = TRUE;
      break;
    case 'q' : //Quite
      verbose = FALSE;
      break;
    default :
      die("Internal argument parsing error");
    }
  }
  
  data = (double *) calloc(MAX_K, sizeof(double));
  mean_prior = (double *) calloc(MAX_K, sizeof(double));


  if(verbose) verbose_stream = out_stream;

  fprintf(out_stream, "# Optimal Posterior (%s)\n\n", VERSION);
  read_data(in_stream, &K, data, mean_prior );
  if(verbose) print_ln_prob( K, data, mean_prior, verbose_stream);
  beta = dirichlet_beta_ml(K, data, mean_prior, verbose_stream);
  if(verbose) relative_ent(out_stream, K, data, mean_prior);
  write_data(out_stream, K, data, mean_prior, beta); 
  
  return 0; // Done
}


























